import express from "express";
import { default as makeWASocket, useMultiFileAuthState } from "bail-lite";
import QRCode from "qrcode";
import fs from "fs";

const app = express();
app.use(express.json({ limit: "50mb" }));

let sock;
let qrCodeBase64 = "";
let isConnected = false;

// =========================
//     START WHATSAPP
// =========================
async function startWA() {
    const { state, saveCreds } = await useMultiFileAuthState("./auth");

    sock = makeWASocket({
        printQRInTerminal: true,
        auth: state,
        syncFullHistory: false,
    });

    // Events
    sock.ev.on("connection.update", async (update) => {
        const { connection, qr } = update;

        if (qr) {
            qrCodeBase64 = await QRCode.toDataURL(qr);
            isConnected = false;
            console.log("📡 QR Generated (scan it)");
        }

        if (connection === "open") {
            isConnected = true;
            console.log("✅ WhatsApp Connected");
        }

        if (connection === "close") {
            console.log("❌ Disconnected. Reconnecting…");
            startWA();
        }
    });

    sock.ev.on("creds.update", saveCreds);
}

startWA();

// =========================
//   API ROUTES
// =========================

// GET QR Code
app.get("/qr", (req, res) => {
    if (!qrCodeBase64) return res.send("QR not generated yet");
    res.send(`<img src="${qrCodeBase64}" style="width:300px">`);
});

// Check Connection
app.get("/status", (req, res) => {
    res.json({
        status: isConnected ? "connected" : "not_connected",
    });
});

// SEND TEXT MESSAGE
app.post("/send-message", async (req, res) => {
    try {
        const { number, message } = req.body;
        const jid = number + "@s.whatsapp.net";

        await sock.sendMessage(jid, { text: message });

        res.json({ success: true, message: "Message sent" });
    } catch (err) {
        res.json({ success: false, error: err.message });
    }
});

// SEND MEDIA (image/video/document)
app.post("/send-media", async (req, res) => {
    try {
        const { number, caption, file_name, media_base64 } = req.body;

        const buffer = Buffer.from(media_base64, "base64");
        const jid = number + "@s.whatsapp.net";

        await sock.sendMessage(jid, {
            document: buffer,
            mimetype: "application/octet-stream",
            fileName: file_name,
            caption: caption
        });

        res.json({ success: true, message: "Media sent" });
    } catch (err) {
        res.json({ success: false, error: err.message });
    }
});

// =========================
//      START SERVER
// =========================

const PORT = process.env.PORT || 3000;

app.listen(PORT, () => {
    console.log("🚀 Server running on port " + PORT);
});
